{$S-,R-,I-,V-,B-}

  {*********************************************************}
  {*                    TPTSR.PAS 4.03                     *}
  {*        Copyright (c) TurboPower Software 1987.        *}
  {* Portions copyright (c) Sunny Hill Software 1985, 1986 *}
  {*     and used under license to TurboPower Software     *}
  {*                 All rights reserved.                  *}
  {*********************************************************}

unit TpTsr;
  {-TSR management routines}

interface

uses
  Dos,                       {Turbo's DOS/BIOS routines}
  TpInt;                     {ISR management routines}

  {$DEFINE ThwartSideKick}   {if DEFINEd, TPTSR tries to thwart SideKick}

type
  IfcPtr = ^IfcRecord;
  IfcRecord = record         {** don't change order **}
                NamePtr : ^string;
                Version : Word;
                CmdEntryPtr : Pointer;
                PrevIfc,
                NextIfc : IfcPtr;
              end;
const
  MaxPopups = 8;             {maximum number of popup routines}
  SideKickLoaded : Boolean = False; {if True, SideKick is loaded}
  Int5Handle = 1;            {ISR handles used by our interrupt handlers}
  Int8Handle = 2;
  Int9Handle = 3;
  Int10Handle = 4;
  Int13Handle = 5;
  Int14Handle = 6;
  Int16Handle = 7;
  Int17Handle = 8;
  Int25Handle = 9;
  Int26Handle = 10;
  Int28Handle = 11;
var
  ThisIfc : IfcRecord;

function ParagraphsToKeep : Word;
  {-Returns #. of paragraphs currently used for code, data, stack, and heap.}

function TerminateAndStayResident(ParasToKeep : Word; ExitCode : Byte) : Boolean;
  {-Terminate and stay resident, freeing up all our memory except ParasToKeep.
    Returns False if unable to release memory.}

procedure PopupsOn;
  {-Turns popups on}

procedure PopupsOff;
  {-Turns popups off}

function DefinePopProc(var Handle : Byte; Routine, StackPtr : Pointer) : Boolean;
  {-Defines a routine that can be triggered by calling SetPopTicker. Returns
    false if no more Popup handles are available.}

procedure DeletePopProc(Handle : Byte);
  {-Deletes a popup routine defined with DefinePopProc.  Its handle may then
    be used for other popups.}

function DefinePop(HotKey : Word; Routine, StackPtr : Pointer;
                   DosWait : Boolean) : Boolean;
  {-Defines a popup routine associated with a particular hotkey. Returns
    False if no more Popup handles are available.}

procedure DeletePop(HotKey : Word);
  {-Deletes the popup routine associated with HotKey.  Its handle may then
    be used for other popups.}

function ChangeHotKey(OldHotKey, NewHotKey : Word) : Boolean;
  {-Change a popup's hotkey from OldHotKey to NewHotKey}

procedure SetPopTicker(Handle : Byte; TimeOut : Word);
  {-Sets the pop ticker for the indicated procedure, clears interrupts,
    then returns.  No range checking is performed.}

procedure RemovePops;
  {-Removes all popup routines from the system and releases all associated
    interrupt vectors.}

procedure InitPops;
  {-Must be called before any other popup specific routines.}

function INT24Result : Word;
  {-Returns the latest result from the INT $24 handler}

function DisableTSR : Boolean;
  {-Disable TSR by restoring interrupt vectors and releasing memory. This
    does *not* halt the program. Returns false if it's not safe.}

  {* standard interface routines *}

procedure InstallModule(var ModuleName : string; CmdEntryRoutine : Pointer);
  {-Installs this program as a resident module that can be located and
    accessed by other programs.}

procedure UninstallModule;
  {-Uninstalls the module from the linked list of modules.}

function ModulePtrByName(var ModuleName : string) : IfcPtr;
  {-Returns a pointer to the IfcRecord for the module named ModuleName or Nil.}

function ModuleInstalled(var ModuleName : string) : Boolean;
  {-Returns true if ModuleName is installed.}

  {==========================================================================}

implementation

type
  SegOfs = record
             O, S : Word;
           end;
const
  WentResident : Boolean = False;
  IfcInstalled : Boolean = False;
  MaxScanCode = $86;         {highest scan code we can use -- do not change}
  IfcSignature : Word = $0F0F0; {*** do not change!! ***}
  IfcVersion = $400;         {version number of interface}
  IRET : Byte = $CF;
var
  Initialized : Boolean;
  TopOfHeap : Pointer;
  DosVersion : Word;
  SaveExitProc : Pointer;
  SaveSSeg : Word;

  {Addresses of popup routines}
  PopupAddrs : array[1..MaxPopups] of Pointer;

  {Stacks for popup routines}
  PopupStacks : array[1..MaxPopups] of Pointer;

  {Flags for popups in use}
  PopupInUse : array[1..MaxPopups] of Boolean;

  {Flags to indicate popups that need access to DOS}
  DosWaitFlags : array[1..MaxPopups] of Boolean;

  {Table linking scan codes to their associated popups}
  PopupKeys : array[0..MaxScanCode] of Byte;

  {contains the high byte of each popup's hotkey}
  ShiftKeys : array[0..MaxScanCode] of Byte;

  {pointers to variables hidden in TPTSR.OBJ}
  PopTickerPtr : ^Word;      {Points to var. indicating if we are trying to pop up}
  PopupsEnabledPtr : ^Boolean; {Points to flag indicating if popups are enabled}
  PopupToCallPtr : ^Byte;    {Points to var. with the handle of a popup to call}

  function ParagraphsToKeep : Word;
    {-Returns # of paragraphs currently used for code, data, stack, and heap.}
  begin
    {keep everything from PSP to top of heap}
    ParagraphsToKeep := Succ(Seg(HeapPtr^)-PrefixSeg);
  end;

  function HeapEnd : Pointer;
    {-Returns a pointer to the end of the free list}
  begin
    if Ofs(FreePtr^) = 0 then
      {Free list is empty}
      HeapEnd := Ptr(Seg(FreePtr^)+$1000, 0)
    else
      HeapEnd := Ptr(Seg(FreePtr^)+Ofs(FreePtr^) shr 4, Ofs(FreePtr^) and $F);
  end;

  function PtrDiff(H, L : Pointer) : LongInt;
    {-Return the number of bytes between H^ and L^. H is the higher address}
  var
    High : SegOfs absolute H;
    Low : SegOfs absolute L;
  begin
    PtrDiff := (LongInt(High.S) shl 4+High.O)-(LongInt(Low.S) shl 4+Low.O);
  end;

  function TerminateAndStayResident(ParasToKeep : Word; ExitCode : Byte) : Boolean;
    {-Terminate and stay resident, freeing up all memory except ParasToKeep.
      Returns False if unable to release memory.}
  var
    TopOfMem,
    SaveFreePtr,
    NewFreePtr : Pointer;
    MaxParas,
    NewFreeSeg,
    FreeListSize : Word;
    Regs : Registers;
  begin
    {if we return from this call, we failed to go resident}
    TerminateAndStayResident := False;

    {exit if SideKick is loaded and couldn't be thwarted}
    if SideKickLoaded then
      Exit;

    {reject requests for less than the bare minimum or more than the maximum}
    TopOfMem := Ptr(PrefixSeg, 2);
    MaxParas := Word(TopOfMem^)-PrefixSeg;
    if (ParasToKeep < ParagraphsToKeep) or (ParasToKeep > MaxParas) then
      Exit;

    {Calculate the number of bytes to move}
    FreeListSize := PtrDiff(TopOfHeap, HeapEnd);

    {save the current value of FreePtr}
    SaveFreePtr := FreePtr;

    {calculate new location of the free list}
    NewFreeSeg := (PrefixSeg+ParasToKeep)-$1000;
    if FreeListSize = 0 then
      NewFreePtr := Ptr(NewFreeSeg, 0)
    else
      NewFreePtr := Ptr(NewFreeSeg, Ofs(FreePtr^));

    {Move the free list down}
    if FreeListSize <> 0 then
      Move(FreePtr^, NewFreePtr^, FreeListSize);
    FreePtr := NewFreePtr;

    {set flag for error/exit handler}
    WentResident := True;

    {restore vectors captured by Turbo}
    SetIntVec($00, SaveInt00);
    SetIntVec($02, SaveInt02);
    SetIntVec($23, SaveInt23);
    SetIntVec($24, SaveInt24);
    SetIntVec($75, SaveInt75);

    {go resident}
    Regs.AH := $31;          {DOS Keep function}
    Regs.AL := ExitCode;     {return code for DOS}
    Regs.DX := ParasToKeep;  {paragraphs to keep}
    MsDos(Regs);

    {if we get this far, we failed, so restore FreePtr}
    FreePtr := SaveFreePtr;
    WentResident := False;
  end;

  procedure PopupsOn;
    {-Turns popups on}
  begin
    PopupsEnabledPtr^ := True;
  end;

  procedure PopupsOff;
    {-Turns popups off}
  begin
    PopupsEnabledPtr^ := False;
  end;

  function ChangeHotKey(OldHotKey, NewHotKey : Word) : Boolean;
    {-Change a hotkey from OldHotKey to NewHotKey}
  var
    Handle : Byte;
    OldScan : Byte absolute OldHotKey;
    NewScan : Byte absolute NewHotKey;
  begin
    ChangeHotKey := False;
    case Lo(OldHotKey) of
      1..MaxScanCode :
        if (PopupKeys[OldScan] <> 0) and
          ((OldScan = NewScan) or (PopupKeys[NewScan] = 0)) then begin
          {with interrupts off, switch hot keys}
          InterruptsOff;

          {get rid of the current assignment}
          Handle := PopupKeys[OldScan];
          PopupKeys[OldScan] := 0;
          ShiftKeys[OldScan] := 0;

          PopupKeys[NewScan] := Handle;
          ShiftKeys[NewScan] := Hi(NewHotKey);
          InterruptsOn;
          ChangeHotKey := True;
        end;
    end;
  end;

  procedure DeletePop(HotKey : Word);
    {-Deletes the popup routine associated with HotKey.  Its handle may then
      be used for other popups.}
  begin
    case Lo(HotKey) of
      1..MaxScanCode :
        if (PopupKeys[Lo(HotKey)] <> 0) then begin
          {with interrupts off, disable the popup}
          InterruptsOff;
          PopupAddrs[PopupKeys[Lo(HotKey)]] := nil;
          PopupKeys[Lo(HotKey)] := 0;
          ShiftKeys[Lo(HotKey)] := 0;
          InterruptsOn;
        end;
    end;
  end;

  function DefinePopProc(var Handle : Byte; Routine, StackPtr : Pointer) : Boolean;
    {-Defines a routine that can be triggered by calling SetPopTicker. Returns
      false if no more Popup handles are available.}
  var
    B : Byte;
  begin
    {search for an available handle}
    for B := 1 to MaxPopups do
      if (PopupAddrs[B] = nil) then begin
        {assign the handle}
        Handle := B;

        {install the popup}
        PopupAddrs[B] := Routine;
        PopupStacks[B] := StackPtr;
        PopupInUse[B] := False;

        {assume that waiting is necessary}
        DosWaitFlags[B] := True;

        {signal success}
        DefinePopProc := True;
        Exit;
      end;

    {if we get to here we failed}
    DefinePopProc := False;
  end;

  procedure DeletePopProc(Handle : Byte);
    {-Deletes a popup routine defined with DefinePopProc.  Its handle may then
      be used for other popups.}
  begin
    case Handle of
      1..MaxPopups : PopupAddrs[Handle] := nil;
    end;
  end;

  function DefinePop(HotKey : Word; Routine, StackPtr : Pointer;
                     DosWait : Boolean) : Boolean;
    {-Defines a popup routine associated with a particular hotkey. Returns
      False if no more Popup handles are available.}
  var
    B : Byte;
  begin
    {assume failure}
    DefinePop := False;

    case Lo(HotKey) of
      1..MaxScanCode :
        if PopupKeys[Lo(HotKey)] = 0 then
          if DefinePopProc(B, Routine, StackPtr) then begin
            {set the DOS wait flag}
            DosWaitFlags[B] := DosWait;

            {set the shift key}
            ShiftKeys[Lo(HotKey)] := Hi(HotKey);

            {If popups are on, next statement makes this one active}
            PopupKeys[Lo(HotKey)] := B;
            DefinePop := True;
          end;
    end;
  end;

  procedure SetPopTicker(Handle : Byte; TimeOut : Word);
    {-Sets the pop ticker for the indicated procedure, clears interrupts,
      then returns.  No range checking is performed.}
  begin
    if PopupsEnabledPtr^ then
      if (PopupAddrs[Handle] <> nil) and not PopupInUse[Handle] then begin
        {Turn interrupts off}
        InterruptsOff;

        if PopTickerPtr^ <> 0 then
          PopupInUse[PopupToCallPtr^] := False;
        PopupInUse[Handle] := True;
        PopupToCallPtr^ := Handle;
        PopTickerPtr^ := TimeOut;
      end;
  end;

  procedure RemovePops;
    {-Disables all popups and releases all interrupt vectors.}
  begin
    PopupsOff;
    RestoreAllVectors;
    Initialized := False;
  end;

  {$L TPTSR.OBJ}

  {see TPTSR.ASM for documentation}
  procedure Int9;  external;
  procedure Int28; external;
  procedure Int8;  external;
  procedure Int5;  external;
  procedure Int10; external;
  procedure Int13; external;
  procedure Int14; external;
  procedure Int16; external;
  procedure Int17; external;
  procedure Int25; external;
  procedure Int26; external;

  function IoResultPrim : Byte;
    {-Calls IoResult for Int24Result}
  begin
    IoResultPrim := IoResult;
  end;

  function INT24Result : Word;
    external {TPTSR} ;

  function InitTsrPtrs : Boolean;
    {-Initializes pointers to hidden variables and pointers that indicate when
      DOS is active. Returns false if unsupported version of DOS is found.}
    external {TPTSR} ;

  procedure InitPops;
    {-Must be called before any other popup specific routines.}

  {$IFDEF ThwartSideKick}
  type
    Array5 = array[0..4] of Char;
  {$ENDIF}

  const
    {                I B K S}
    SideKickFlag = $49424B53;

    {$IFDEF ThwartSideKick}
    SideKickCode : Array5 = (
      #$1F,        {POP  DS}
      #$8C, #$CB,  {MOV  BX,CS}
      #$3B, #$C3); {CMP  AX,BX}
     {#$74, #$0B    JZ   xxxx <-- if found, change the JZ to a JMP SHORT}
    {$ENDIF}

  var
    P : Pointer;
    PSO : SegOfs absolute P;
    {$IFDEF ThwartSideKick}
      PA5 : ^Array5;
      I : Word;
    {$ENDIF}

    procedure InitError(WrongDos : Boolean);
      {-Initialization error}
    begin
      if WrongDos then
        WriteLn('Unsupported version of DOS')
      else
        WriteLn('Unable to install TSR manager');
      Halt(1);
    end;

  begin
    {don't do this twice}
    if Initialized then
      Exit;

    {test for SideKick: 'SKBI' just before start of INT $8 handler}
    GetIntVec($08, P);
    SideKickLoaded := (MemL[PSO.S:PSO.O-4] = SideKickFlag);

    {$IFDEF ThwartSideKick}
    if SideKickLoaded then begin
      {see if we can thwart its INT 9 grabber so we can go resident}
      PA5 := P;
      {search for the tell-tale code}
      for I := 0 to 100 do begin
        {have we found it?}
        if PA5^ = SideKickCode then begin
          {if so, disable the INT 9 grabber...}
          Inc(Word(PA5), 5);
          {make sure it's OK to change it}
          if (PA5^[0] = #$74) or (PA5^[0] = #$EB) then begin
            InterruptsOff;
            PA5^[0] := #$EB; {turn the conditional jump into a short jump}
            InterruptsOn;

            {...and clear the SideKick flag}
            SideKickLoaded := False;
          end;
        end;
        Inc(Word(PA5));
      end;
    end;
    {$ENDIF}

    {Initialize our tables}
    FillChar(PopupAddrs, SizeOf(PopupAddrs), 0);
    FillChar(PopupStacks, SizeOf(PopupStacks), 0);
    FillChar(PopupInUse, SizeOf(PopupInUse), 0);
    FillChar(PopupKeys, SizeOf(PopupKeys), 0);
    FillChar(ShiftKeys, SizeOf(ShiftKeys), 0);
    FillChar(DosWaitFlags, SizeOf(DosWaitFlags), 0);

    {halt if unsupported DOS version found}
    if not InitTsrPtrs then
      InitError(True);

    {initialize our interrupt vectors}
    if not InitVector($5, Int5Handle, @Int5) then
      InitError(False);
    if not InitVector($8, Int8Handle, @Int8) then
      InitError(False);
    if not InitVector($9, Int9Handle, @Int9) then
      InitError(False);
    if not InitVector($13, Int13Handle, @Int13) then
      InitError(False);
    if not InitVector($16, Int16Handle, @Int16) then
      InitError(False);
    if not InitVector($25, Int25Handle, @Int25) then
      InitError(False);
    if not InitVector($26, Int26Handle, @Int26) then
      InitError(False);
    if not InitVector($28, Int28Handle, @Int28) then
      InitError(False);

    {don't filter the following interrupts if we're running DOS 2.x or 3.x}
    if (DosVersion >= $4000) then begin
      if not InitVector($10, Int10Handle, @Int10) then
        InitError(False);
      if not InitVector($14, Int14Handle, @Int14) then
        InitError(False);
      if not InitVector($17, Int17Handle, @Int17) then
        InitError(False);
    end;

    Initialized := True;
  end;

  procedure InstallModule(var ModuleName : string; CmdEntryRoutine : Pointer);
    {-Installs this program as a resident module that can be located and
      accessed by other programs.}
  var
    P : IfcPtr;
    Regs : Registers;
  begin
    if not IfcInstalled then
      with ThisIfc, Regs do begin
        {see if anyone else is home}
        AX := IfcSignature;
        Intr($16, Regs);

        {if AX comes back with its bits flipped, we're not alone}
        if AX = not Word(IfcSignature) then begin
          P := Ptr(ES, DI);
          P^.NextIfc := @ThisIfc;
          PrevIfc := P;
        end
        else
          PrevIfc := nil;

        {initialize the other fields in the record}
        NamePtr := @ModuleName;
        NextIfc := nil;
        Version := IfcVersion;
        if CmdEntryRoutine = nil then
          CmdEntryPtr := @IRET
        else
          CmdEntryPtr := CmdEntryRoutine;

        IfcInstalled := True;
      end;
  end;

  procedure UninstallModule;
    {-Uninstalls the module from the linked list of modules.}
  begin
    if IfcInstalled then
      with ThisIfc do begin
        {fix the linked list of modules}
        if PrevIfc <> nil then
          PrevIfc^.NextIfc := NextIfc;
        if NextIfc <> nil then
          NextIfc^.PrevIfc := PrevIfc;
        IfcInstalled := False;
      end;
  end;

  function ModulePtrByName(var ModuleName : string) : IfcPtr;
    {-Returns a pointer to the IfcRecord for the module named ModuleName or Nil.}
  var
    P : IfcPtr;
    FoundIfc : Boolean;
  begin
    {assume failure}
    ModulePtrByName := nil;
    FoundIfc := False;

    {since Intr() is not re-entrant, and this might be called from inside a
     popup, use inline}
    inline(
      $A1/>IfcSignature/     {mov ax,[>IfcSignature] ;standard interface function code}
      $50/                   {push ax                ;save this}
      $CD/$16/               {int $16                ;call INT $16}
      $5B/                   {pop bx                 ;BX = IfcSignature}
      $F7/$D3/               {not bx                 ;BX = not IfcSignature}
      $39/$D8/               {cmp ax,bx              ;ax = not IfcSignature?}
      $75/$0A/               {jne NotFound           ;Ifc handler not found?}
      $C6/$46/<FoundIfc/$01/ {mov [bp+<FoundIfc],1   ;set Found flag}
      $89/$7E/<P/            {mov [bp+<P],di         ;offset of list pointer in P}
      $8C/$46/<P+2);         {mov [bp+<P+2],es       ;segment of list pointer in P}
                             {NotFound:}

    {if AX didn't come back with its bits flipped, no modules are installed}
    if not FoundIfc then
      Exit;

    {search backward through the list -- P already initialized}
    while (P <> nil) do
      with P^ do
        if NamePtr^ = ModuleName then begin
          ModulePtrByName := P;
          Exit;
        end
        else
          P := P^.PrevIfc;
  end;

  function ModuleInstalled(var ModuleName : string) : Boolean;
    {-Returns true if ModuleName is installed.}
  begin
    {let ModulePtrByName do the searching}
    ModuleInstalled := ModulePtrByName(ModuleName) <> nil;
  end;

  function DisableTSR : Boolean;
    {-Disable TSR by restoring interrupt vectors and releasing memory. This
      does *not* halt the program. Returns false if it's not safe.}
  var
    Regs : Registers;
    Safe : Boolean;
    I : Word;
    P : Pointer;
    VecsGrabbed : set of Byte;
  begin
    {see if it's safe to disable the TSR}
    Safe := True;

    {initialize VecsGrabbed entries to false}
    FillChar(VecsGrabbed, SizeOf(VecsGrabbed), 0);

    {go backwards, looking for changed vectors that have been grabbed only once}
    for I := MaxIsrs downto 1 do
      with ISR_Array[I] do
        if Captured then begin
          GetIntVec(IntNum, P);
          if (P <> NewAddr) and not(IntNum in VecsGrabbed) then
            Safe := False;
          VecsGrabbed := VecsGrabbed+[Lo(IntNum)];
        end;
    {^----- this doesn't always work if the same ISR has been grabbed twice}

    {don't disable if it's not safe}
    DisableTSR := Safe;
    if not Safe then
      Exit;

    {disable popups and restore any other vectors taken over}
    PopupsOff;
    Initialized := False;
    RestoreAllVectors;
    UninstallModule;

    with Regs do begin
      {close all file handles, ignoring error codes}
      for I := 0 to 19 do begin
        AH := $3E;           {close file handle function}
        BX := I;             {handle}
        MsDos(Regs);
      end;

      {release environment}
      AH := $49;             {free allocated memory function}
      ES := MemW[PrefixSeg:$2C]; {segment of environment from PSP}
      MsDos(Regs);

      {release program's memory}
      AH := $49;
      ES := PrefixSeg;
      MsDos(Regs);
    end;
  end;

  procedure EmergencyExit;
    {-Called by exit/error handler in case of runtime error while popped up}
    external {TPTSR} ;

  {$F+}
  procedure TpTsrExit;
    {-Error/exit handler}
  begin
    {restore previous exit handler}
    ExitProc := SaveExitProc;

    {uninstall the module and turn popups off}
    UninstallModule;
    PopupsOff;

    {if we're already resident, then WHAT?}
    if WentResident then begin
      {try to disable the TSR}
      if DisableTSR then
        {if an alternate stack is in use, try to exit semi-gracefully by
         calling an assembly language routine that tries to salvage the system}
        if SSeg <> SaveSSeg then
          EmergencyExit;
      {else, throw up our hands and hope for the best, because our stack
         has surely been trashed}
    end;
  end;
  {$F-}

begin
  Initialized := False;
  FillChar(ThisIfc, SizeOf(ThisIfc), 0);

  {Save top of heap for later}
  TopOfHeap := HeapEnd;

  {save SSeg for later}
  SaveSSeg := SSeg;

  {prepare for popups}
  InitPops;

  {set up error/exit handler}
  SaveExitProc := ExitProc;
  ExitProc := @TpTsrExit;
end.
